# Implementation Plan - Action Dispatcher System (Refactor)

## Goal
Replace the fragile function-hook history system with a robust **Action Dispatcher (Command Pattern)**. This ensures that EVERY state change is explicitly registered, executed, and recorded in history, addressing the user's issue that "nothing changes" on the slider.

## The Problem with Current Approach
- Function hooks (`old = fn; fn = new`) can be unreliable if references are captured early (e.g., by event listeners created before the hook).
- State updates might be happening *outside* the tracked functions.
- Debugging is difficult.

## The Solution: Centralized Dispatcher
We will implement a `dispatch(action, payload)` system.
1.  **Action**: A string identifier (e.g., `'OPEN_SUBLEVEL'`).
2.  **Payload**: Data needed for the action (e.g., `{ id: 'sub1' }`).
3.  **Flow**: `UI Event` -> `dispatch()` -> `Execute Logic` -> `Record History` -> `Update UI`.

## Proposed Changes

### `gra.html`

#### 1. [NEW] `GameActions` Registry
Define all valid game mutations in one place:
```javascript
const GameActions = {
  OPEN_SUBLEVEL: (payload) => {
    openedSublevels.add(payload.id);
    // ... logic ...
  },
  PICKUP_ITEM: (payload) => {
    inventory.push(payload.item);
    // ... remove from source ...
  },
  LEARN_SKILL: (payload) => {
    skills.push(payload.skill);
    // ... remove from source ...
  },
  DROP_ITEM: (payload) => {
    inventory.splice(payload.index, 1);
  }
};
```

#### 2. [NEW] `dispatch(actionType, payload)` Function
The single entry point for all changes:
```javascript
function dispatch(type, payload) {
  console.log('ACTION:', type, payload); // Visible proof for user
  
  const action = GameActions[type];
  if (!action) return console.error('Unknown action:', type);

  // 1. Execute Action
  action(payload);

  // 2. Record History
  history.push();

  // 3. Save & Render
  saveState(); // localStorage
  renderAll(); // Refresh UI
}
```

#### 3. [MODIFY] UI Event Listeners
Update `renderLevels`, `renderInv`, `renderSkills` to call `dispatch` instead of direct functions.
- **Before**: `onclick = () => toggleSublevel(...)`
- **After**: `onclick = () => dispatch('OPEN_SUBLEVEL', { id: ... })`

#### 4. [MODIFY] `HistoryManager`
Simplify it. It no longer needs to hook functions. It just snapshots state when `dispatch` calls `push()`.

## Verification Plan
1.  **Click Open Sublevel**:
    - Check Console: Should see `ACTION: OPEN_SUBLEVEL`.
    - Check Slider: Should move +1.
    - Check UI: Lock icon changes to Open.
2.  **Undo**:
    - Press Ctrl+Z.
    - Check UI: Sublevel closes. Slider moves -1.
3.  **Redo**:
    - Press Ctrl+Y.
    - Check UI: Sublevel opens. Slider moves +1.

This approach guarantees that if the action happens, it IS recorded.
